#pragma warning ( 3 : 4706)
/********************************* Game.cpp ***********************************\
* This module contains all the base functions used for the game code.  This    *
* includes setup, teardown, and the main game loop.                            *
\******************************************************************************/

#include <windows.h>
#include <commdlg.h>
#include <list>
#include <vector>
#include <math.h>
using std::list;
using std::vector;

#include "WinApi.h"
#include "Externs.h"
#include "Defines.h"
#include "IntrFace\CDSound.h"
#include "IntrFace\CGraphic.h"
#include "Binfile\binfmem.h"
#include "Binfile\binfstd.h"
#include "Game.h"
#include "Init.h"
#include "Debug.h"
#include "Graphics\PrimDraw.h"
#include "VectObject.h"
#include "Resource.h"

// the game state is updated 10 times per second
#define VFPS 20

CGraphics   Cgraph;
CDSound     Cdsound;
CDInput     Cdinput;
CMidas      Cmidas;
CDIDevice   Cdmouse;
CDIDevice   Cdkeyboard;

volatile BOOL g_bNeedRedraw;

static Status UpdateGameState();
static Status UpdateScreen();
static int    LoadGameData();
static int    UnloadGameData();

CDDSurface *BackBuffer, *FakeBack;

enum Ops { opPOINT, opLINE, opCIRCLE, opLINELIST, opBOUNDS };
static int   ObjectNumber;
static Ops cur_op = opPOINT;
static char *OpNames[] = { "POINT", "LINE", "CIRCLE", "LINELIST", "BOUNDS" };
static double Scale=1.0;
static Graphics::PrimDraw PD;
static int ShowGrid=2;
bool InOp;
static RECT OpRect;
static int Buffer[500], CurIndex;
static int mousex, mousey;
static int Xoff, Yoff;
static COLORREF Customs[16];

#define MIN(a,b) ((a) < (b) ? (a) : (b))
#define MAX(a,b) ((a) > (b) ? (a) : (b))
#define REVERSE(c) ( ((c)>>16) | ((c) & 0x00FF00) | ( ((c) & 0xFF) << 16) )
#define TO16BPP(c) ( (((c)>>19)<<11) | (((((c)>>8)&0xFF)/4)<<5) | (((c) & 0xFF)/8) )
#define TO24BPP(c) ( (((c)>>11)<<19) | ((((c)>>5)&0x3F)<<10) | (((c)&0x1F)<<3) )
struct sObject
{
   sObject() { name[0] = 0; };
   Vectors::VectObject obj;
   char name[128];
};

struct Point
{
   Point() { };
   Point(int a,int b) : x(a),y(b) { };
   bool operator==(Point p) { return p.x == x && p.y == y; };
   int x,y;
};

vector<Point> Points;
vector<int> Indices;

vector<sObject*> Objs(1);
CDDColor Color = MYRGB(255,255,255);

OPENFILENAME Ofn;

/* Called VFPS times per second. Update the game state here.
   Return G_UPDATED if you make a change which requires a screen update
   or G_RUNNING if not
*/
Status UpdateGameState()
{
   static Graphics::Primitive *Cur_Prim;
   static Graphics::PrimFactory PF;
   static mbinfile MemFile;
   static int Ticks;
   static char FileName[256];

   int r_mousex, r_mousey;
   Status ret = G_RUNNING;


   if (!Ticks)
   {
      MemFile.open(Buffer, 500*sizeof(int), MemFile.openro);
      Ofn.lpstrFile = FileName;
      Ticks++;
   }
   
   r_mousex = MouseX-Xoff, r_mousey = MouseY-Yoff;
   if (mousex != (r_mousex-320)/Scale || mousey != (r_mousey-240)/Scale)
   {
      char buf[32];
      wsprintf(buf, "%d, %d", mousex = (r_mousex-320)/Scale + (r_mousex >= 320 ? 0.5 : -0.5), mousey = (r_mousey-240)/Scale + (r_mousey >= 240 ? 0.5 : -0.5));
      SetWindowText(Child[STC_COORDS], buf);

      if (CurIndex && (cur_op == opCIRCLE || cur_op == opBOUNDS))
      {
         char buf[40];
         wsprintf(buf, "Radius: %d", (int)sqrt((Buffer[0]-mousex)*(Buffer[0]-mousex) + (Buffer[1]-mousey)*(Buffer[1]-mousey)));
         SetWindowText(Child[STC_STATUS], buf);
         ret = G_UPDATED;
      }
      if (Cur_Prim) ret = G_UPDATED;
   }

   if(g_bClicked || g_bRclick)
   {
      bool Valid = false;
      if (r_mousex >= 0 && r_mousex < 640 && r_mousey >= 0 && r_mousey < 480) Valid = true;

      switch(cur_op)
      {
         case opPOINT:
            if (!g_bClicked) break;
            if (Valid)
            {
               Cur_Prim = PF.New(Graphics::PT_POINT);
               Buffer[0] = mousex, Buffer[1]=mousey;
               MemFile.seek(0);
               Cur_Prim->Load(&MemFile);
               Objs[ObjectNumber]->obj.Prims.push_back(Cur_Prim);
               Objs[ObjectNumber]->obj.Colors.push_back(Color);
               Cur_Prim = 0;
               ret = G_UPDATED;
            }
            break;

         case opLINE:
            if (!g_bClicked) break;
            if (Valid)
            {
               if (!CurIndex)
               {
                  Cur_Prim = PF.New(Graphics::PT_LINE);
                  Buffer[0] = mousex, Buffer[1] = mousey;
                  CurIndex = 2;
                  ShowCursor(FALSE);
                  InOp = true;
               }
               else
               {
                  Buffer[2] = mousex, Buffer[3] = mousey;
                  MemFile.seek(0);
                  Cur_Prim->Load(&MemFile);
                  Objs[ObjectNumber]->obj.Prims.push_back(Cur_Prim);
                  Objs[ObjectNumber]->obj.Colors.push_back(Color);
                  Cur_Prim = 0;
                  CurIndex = 0;
                  ShowCursor(TRUE);
                  InOp = false;
                  ret = G_UPDATED;
               }
            }
            break;
         
         case opCIRCLE:
            if (!g_bClicked) break;
            if (Valid)
            {
               if (!CurIndex)
               {
                  Cur_Prim = PF.New(Graphics::PT_CIRCLE);
                  Buffer[0] = mousex, Buffer[1] = mousey;
                  CurIndex=2;
                  ShowCursor(FALSE);
                  InOp = true;
               }
               else
               {
                  Buffer[2] = sqrt((Buffer[0]-mousex)*(Buffer[0]-mousex) + (Buffer[1]-mousey)*(Buffer[1]-mousey));
                  MemFile.seek(0);
                  Cur_Prim->Load(&MemFile);
                  Objs[ObjectNumber]->obj.Prims.push_back(Cur_Prim);
                  Objs[ObjectNumber]->obj.Colors.push_back(Color);
                  Cur_Prim = 0;
                  CurIndex = 0;
                  ShowCursor(TRUE);
                  InOp = false;
                  ret = G_UPDATED;
               }
            }
            break;

         case opBOUNDS:
            if(!g_bClicked) break;
            if(Valid)
            {
               if(!CurIndex)
               {
                  Buffer[0] = Buffer[1] = 0;
                  CurIndex = 2;
                  ShowCursor(FALSE);
                  InOp = true;
               }
               else
               {
                  Objs[ObjectNumber]->obj.Bounds = sqrt(mousex*mousex + mousey*mousey);
                  CurIndex = 0;
                  ShowCursor(TRUE);
                  InOp = false;
                  ret = G_UPDATED;
               }
            }
            break;

         case opLINELIST:
            if(!CurIndex && Valid && g_bClicked)
            {
               Cur_Prim = PF.New(Graphics::PT_LINELIST);
               ShowCursor(FALSE);
               InOp = true;
               Points.resize(1);
               Indices.resize(1);
               Points[0] = Point(mousex,mousey);
               Indices[0] = 0;
               CurIndex = 1;
               g_bRclick = false;
            }
            else if(CurIndex && Valid && g_bClicked)
            {
               Point p(mousex, mousey);
               int i, num = Points.size();
               for(i=0;i<num;i++) if(Points[i] == p) break;
               if (i==num) Points.resize(i+1);
               Points[i] = p;
               if (Indices.size() == CurIndex) Indices.resize(CurIndex+1);
               Indices[CurIndex++] = i;
            }
            else if(CurIndex && (!Valid || g_bRclick))
            {
               Point p;
               int i, num;
               if (Indices.size() == 1)
               {
                  delete Cur_Prim;
                  Cur_Prim = 0;
                  CurIndex = 0;
                  InOp = false;
                  ShowCursor(TRUE);
                  ret = G_UPDATED;
                  g_bRclick = false;
                  break;
               }
               if(g_bRclick && Indices[CurIndex-1] != 0)
               {
                  Indices.resize(CurIndex+1);
                  Indices[CurIndex] = 0;
                  g_bRclick = false;
               }
               Buffer[0] = num = Points.size();
               CurIndex = 1;
               for(i=0;i<num;i++)
               {
                  p = Points[i];
                  Buffer[CurIndex++] = p.x;
                  Buffer[CurIndex++] = p.y;
               }
               Buffer[CurIndex++] = num = Indices.size();
               for(i=0;i<num;i++) Buffer[CurIndex++] = Indices[i];
               MemFile.seek(0);
               Cur_Prim->Load(&MemFile);
               Objs[ObjectNumber]->obj.Prims.push_back(Cur_Prim);
               Objs[ObjectNumber]->obj.Colors.push_back(Color);
               Cur_Prim = 0;
               CurIndex = 0;
               ShowCursor(TRUE);
               InOp = false;
               ret = G_UPDATED;
            }
            break;
      }
      g_bClicked = false;
   }

   if(KeyHistory.size())
   {
      KeyPress c = *KeyHistory.begin();
      KeyHistory.pop_front();

      if (c.Down)
      {
         switch(c.Key)
         {
            case DIK_UP:
               Yoff -= 25;
               {
                  char buf[64];
                  wsprintf(buf, "Pos: %d, %d", Xoff, Yoff);
                  SetWindowText(Child[STC_STATUS], buf);
               }
               ret = G_UPDATED;
               break;

            case DIK_DOWN:
               Yoff += 25;
               {
                  char buf[64];
                  wsprintf(buf, "Pos: %d, %d", Xoff, Yoff);
                  SetWindowText(Child[STC_STATUS], buf);
               }
               ret = G_UPDATED;
               break;

            case DIK_LEFT:
               Xoff -= 25;
               {
                  char buf[64];
                  wsprintf(buf, "Pos: %d, %d", Xoff, Yoff);
                  SetWindowText(Child[STC_STATUS], buf);
               }
               ret = G_UPDATED;
               break;

            case DIK_RIGHT:
               Xoff += 25;
               {
                  char buf[64];
                  wsprintf(buf, "Pos: %d, %d", Xoff, Yoff);
                  SetWindowText(Child[STC_STATUS], buf);
               }
               ret = G_UPDATED;
               break;

            case DIK_R:
               Xoff = Yoff = 0;
               Scale = 1.0;
               SetWindowText(Child[STC_STATUS], "Scale/Translation Reset");
               ret = G_UPDATED;
               break;

            case DIK_1:       // switch OP to point
               cur_op = opPOINT;
               ret = G_UPDATED;
               break;

            case DIK_2:       // op to line
               cur_op = opLINE;
               if (Cur_Prim) { delete Cur_Prim; Cur_Prim = 0; }
               CurIndex = 0;
               InOp = false;
               ShowCursor(TRUE);
               ret = G_UPDATED;
               break;

            case DIK_3:       // op to circle
               cur_op = opCIRCLE;
               if (Cur_Prim) { delete Cur_Prim; Cur_Prim = 0; }
               CurIndex = 0;
               InOp = false;
               ShowCursor(TRUE);
               ret = G_UPDATED;
               break;

            case DIK_4:       // op to linelist
               cur_op = opLINELIST;
               if (Cur_Prim) { delete Cur_Prim; Cur_Prim = 0; }
               CurIndex = 0;
               InOp = false;
               ShowCursor(TRUE);
               ret = G_UPDATED;
               break;

            case DIK_5:       // op to bounds
               cur_op = opBOUNDS;
               if (Cur_Prim) { delete Cur_Prim; Cur_Prim = 0; }
               CurIndex = 0;
               InOp = false;
               ShowCursor(TRUE);
               ret = G_UPDATED;
               {
                  char buf[64];
                  wsprintf(buf, "Bounds is %d\n", Objs[ObjectNumber]->obj.Bounds);
                  SetWindowText(Child[STC_STATUS], buf);
               }
               break;

            case DIK_F1:
               MessageBox(NULL,
                          "1 - Op to point\n"
                          "2 - Op to line\n"
                          "3 - Op to circle\n"
                          "4 - Op to linelist\n"
                          "5 - Op to bounding circle\n"
                          ", - Previous object\n"
                          ". - Next object\n"
                          "G - toggle grid\n"
                          "S - Save\n"
                          "L - Load\n"
                          "C - pick color\n"
                          "Ctrl-Z - Undo\n"
                          "Ctrl-D - Delete object\n"
                          "Num + - increase scale\n"
                          "Num - - decrease scale\n"
                          "Arrow Keys - Translate\n"
                          "R - Reset Scale/Pos", "Help", MB_OK);
               break;

            case DIK_COMMA:
               GetWindowText(Child[EDT_NAME], Objs[ObjectNumber]->name, 64);
               if (--ObjectNumber < 0) ObjectNumber = 0;
               SetWindowText(Child[EDT_NAME], Objs[ObjectNumber]->name);
               ret = G_UPDATED;
               break;

            case DIK_PERIOD:
               GetWindowText(Child[EDT_NAME], Objs[ObjectNumber]->name, 64);
               ObjectNumber++;
               if (ObjectNumber == Objs.size())
               {
                  Objs.resize(ObjectNumber+1);
                  Objs[ObjectNumber] = new sObject;
               }
               SetWindowText(Child[EDT_NAME], Objs[ObjectNumber]->name);

               ret = G_UPDATED;
               break;

            case DIK_NUMPADPLUS:
               Scale += Scale >= 5.0 ? 1.0 : 0.5;
               {
                  char buf[32];
                  sprintf(buf, "Scale: %f", Scale);
                  SetWindowText(Child[STC_STATUS], buf);
               }
               ret = G_UPDATED;
               break;

            case DIK_NUMPADMINUS:
               Scale -= Scale >= 5.0 ? 1.0 : 0.5;
               if (Scale < 1.0) Scale = 1.0;
               {
                  char buf[32];
                  sprintf(buf, "Scale: %f", Scale);
                  SetWindowText(Child[STC_STATUS], buf);
               }
               ret = G_UPDATED;
               break;

            case DIK_Z:
               if (KeyboardBuf[DIK_LCONTROL] || KeyboardBuf[DIK_RCONTROL])
               {
                  int l;
                  if (l=Objs[ObjectNumber]->obj.Prims.size())
                  {
                     Objs[ObjectNumber]->obj.Prims.resize(l-1);
                     Objs[ObjectNumber]->obj.Colors.resize(l-1);
                  }
                  ret= G_UPDATED;
               }
               break;

            case DIK_D:
               if (KeyboardBuf[DIK_LCONTROL] || KeyboardBuf[DIK_RCONTROL])
               {
                  int len;
                  if((len=Objs.size()) > 1)
                  {
                     int i;
                     delete Objs[ObjectNumber];
                     for(i=ObjectNumber;i<len-1;i++) Objs[i] = Objs[i+1];
                     if (ObjectNumber == len-1) ObjectNumber--;
                     SetWindowText(Child[EDT_NAME], Objs[ObjectNumber]->name);
                     Objs.resize(len-1);
                  }
                  else if(len == 1)
                  {
                     Objs[0]->obj.Prims.resize(0);
                     Objs[0]->obj.Colors.resize(0);
                     Objs[0]->name[0] = 0;
                     SetWindowText(Child[EDT_NAME], "");
                     KeyPress c;
                     c.Down = TRUE;
                     c.Key = DIK_COMMA;
                     KeyHistory.push_front(c);
                  }
                  else MsgBox("wtf?!");
                  ret=G_UPDATED;
               }
               break;

            case DIK_G:
               if (--ShowGrid < 0) ShowGrid = 2;
               ret=G_UPDATED;
               break;

            case DIK_C:
               {
                  CHOOSECOLOR cc;
                  ZeroMemory(&cc, sizeof(cc));
                  cc.lStructSize = sizeof(cc);
                  cc.hwndOwner = g_hWnd;
                  cc.rgbResult = REVERSE(TO24BPP(Color.c16));
                  cc.lpCustColors = Customs;
                  cc.Flags = CC_FULLOPEN | CC_ANYCOLOR | CC_RGBINIT;
                  if (ChooseColor(&cc)) Color = TO16BPP(REVERSE(cc.rgbResult));
                  char buf[40];
                  wsprintf(buf, "Color is: %d, %d, %d",
                           (Color.c16>>11)*8, ((Color.c16>>5)&0x3f)*4, (Color.c16&0x1F)*8);
                  SetWindowText(Child[STC_STATUS], buf);
               }
               Cdinput.Acquire(&Cdkeyboard);
               break;

            case DIK_S:
               Ofn.Flags = OFN_CREATEPROMPT | OFN_HIDEREADONLY | OFN_NOREADONLYRETURN |
                           OFN_OVERWRITEPROMPT | OFN_PATHMUSTEXIST;
               if (GetSaveFileName(&Ofn))
               {
                  sbinfile file;
                  int         i;
                  char EnumFile[256];

                  if (file.open(FileName, file.openrw | file.opentr))
                  {
                     MsgBox("Error opening file: %s", FileName);
                  }
                  else
                  {
                     int i, num = Objs.size();
                     file.write(&num, sizeof(num));
                     for(i=0;i<num;i++) Objs[i]->obj.Save(&file);
                  }

                  for(i=strlen(FileName);i>0;i--) if (FileName[i] == '\\') break;
                  CopyMemory(EnumFile, FileName, i+1);
                  EnumFile[i+1] = 0;
                  strcat(EnumFile, "Objects.Enum");
                  GetWindowText(Child[EDT_NAME], Objs[ObjectNumber]->name, 64);

                  file.close();
                  if (file.open(EnumFile, file.openrw | file.opentr))
                  {
                     MsgBox("Error opening file: %s", EnumFile);
                  }
                  else
                  {
                     int i, num = Objs.size();
                     char *sep=", ";
                     for(i=0;i<num;i++)
                     {
                        file.write(Objs[i]->name, strlen(Objs[i]->name));
                        file.write(sep, 2);
                     };
                  }
                  file.close();
               }
               Cdinput.Acquire(&Cdkeyboard);
               break;

            case DIK_L:
               Ofn.Flags = OFN_HIDEREADONLY | OFN_PATHMUSTEXIST;
               if (GetOpenFileName(&Ofn))
               {
                  sbinfile file;
                  int         i;
                  char EnumFile[256];

                  if (file.open(FileName, file.openro | file.openex))
                  {
                     MsgBox("Error opening file: %s", FileName);
                  }
                  else
                  {
                     int i, num = Objs.size();
                     for(i=0;i<num;i++) delete Objs[i];
                     file.read(&num, sizeof(num));
                     Objs.resize(num);
                     for(i=0;i<num;i++)
                     {
                        Objs[i] = new sObject;
                        Objs[i]->obj.Load(&file);
                     }
                  }

                  InOp=false;
                  CurIndex=0;
                  ObjectNumber = 0;

                  for(i=strlen(FileName);i>0;i--) if (FileName[i] == '\\') break;
                  CopyMemory(EnumFile, FileName, i+1);
                  EnumFile[i+1] = 0;
                  strcat(EnumFile, "Objects.Enum");

                  file.close();
                  if (file.open(EnumFile, file.openro | file.openex))
                  {
                     MsgBox("Error opening file: %s", EnumFile);
                  }
                  else
                  {
                     int i, num = file.length() + 1;
                     char *list = new char[num], *start, *end;
                     start = list;
                     list[num-1] = 0;
                     file.read(list, num-1);

                     num = Objs.size();

                     for(i=0;i<num;i++)
                     {
                        end = strchr(start, ',');
                        if(end)
                        {
                           memcpy(Objs[i]->name, start, end-start);
                           Objs[i]->name[end-start] = 0;
                           start = end+2;
                        }
                        else strcpy(Objs[i]->name, start);
                     };
                     SetWindowText(Child[EDT_NAME], Objs[0]->name);
                     delete[] list;
                  }
                  file.close();
               }
               Cdinput.Acquire(&Cdkeyboard);
               break;
         }
      }
   }

               
   if (g_bNeedRedraw)
   {
      g_bNeedRedraw = FALSE;
      //Cdinput.Acquire(&Cdmouse);
      Cdinput.Acquire(&Cdkeyboard);
      return G_UPDATED;
   }

   return ret;
} /* UpdateGameState */


Status UpdateScreen()
{
   CDDSurface *myBack;

   // Update the backbuffer here, based on the game state
   SetWindowText(Child[STC_TYPE], OpNames[cur_op]);
   {
      char buf[64];
      wsprintf(buf, "%d of %d", ObjectNumber+1, Objs.size());
      SetWindowText(Child[STC_NUMBER], buf);
   }

   if(!InOp)
   {
      ShowCursor(FALSE);
      Cgraph.ClearSurface(BackBuffer);
      if (ShowGrid)
      {
         if (ShowGrid == 2)
         {
            float i,j;
            for(i=320-Scale*10,j=320+Scale*10;j<640;i-=Scale*10,j+=Scale*10)
            {
               PD.Line(BackBuffer, i+Xoff, 0+Yoff, i+Xoff, 479+Yoff, MYRGB(48, 48, 48));
               PD.Line(BackBuffer, j+Xoff, 0+Yoff, j+Xoff, 479+Yoff, MYRGB(48, 48, 48));
            }
            for(i=240-Scale*10,j=240+Scale*10;j<480;i-=Scale*10,j+=Scale*10)
            {  
               PD.Line(BackBuffer, 0+Xoff, i+Yoff, 639+Xoff, i+Yoff, MYRGB(48, 48, 48));
               PD.Line(BackBuffer, 0+Xoff, j+Yoff, 639+Xoff, j+Yoff, MYRGB(48, 48, 48));
            }
            PD.Line(BackBuffer, 320+Xoff, 0+Yoff, 320+Xoff, 479+Yoff, MYRGB(255, 255, 255));
            PD.Line(BackBuffer, 0+Xoff, 240+Yoff, 649+Xoff, 240+Yoff, MYRGB(255, 255, 255));
         }
         else
         {
            PD.Line(BackBuffer, 320+Xoff, 0+Yoff, 320+Xoff, 479+Yoff, MYRGB(48, 48, 48));
            PD.Line(BackBuffer, 0+Xoff, 240+Yoff, 649+Xoff, 240+Yoff, MYRGB(48, 48, 48));
         }
            
      }
      Objs[ObjectNumber]->obj.Scale(Scale);
      Objs[ObjectNumber]->obj.Draw(&PD, BackBuffer, 320+Xoff, 240+Yoff, 0, 0);
      myBack = BackBuffer;
   }

   if (InOp)
   {
      Cgraph.BlitFast(FakeBack, BackBuffer, 0, 0, NULL);
      switch(cur_op)
      {
         case opLINE:
            PD.Line(FakeBack, Buffer[0]*Scale+320+Xoff, Buffer[1]*Scale+240+Yoff, mousex*Scale+320+Xoff, mousey*Scale+240+Yoff, Color);
            break;

         case opCIRCLE:
            PD.Circle(FakeBack, Buffer[0]*Scale+320+Xoff, Buffer[1]*Scale+240+Yoff, sqrt((Buffer[0]-mousex)*(Buffer[0]-mousex) + (Buffer[1]-mousey)*(Buffer[1]-mousey))*Scale, Color);
            break;

         case opBOUNDS:
            PD.Circle(FakeBack, 320+Xoff, 240+Yoff, sqrt(mousex*mousex + mousey*mousey)*Scale, MYRGB(255, 0, 0));
            break;

         case opLINELIST:
            {
               int i,num=Indices.size();
               if (num > 1)
               {
                  for(i=0;i<num-1;i++) PD.Line(FakeBack, Points[Indices[i]].x*Scale+320+Xoff, Points[Indices[i]].y*Scale+240+Yoff, Points[Indices[i+1]].x*Scale+320+Xoff, Points[Indices[i+1]].y*Scale+240+Yoff, Color);
                  PD.Line(FakeBack, Points[Indices[i]].x*Scale+320+Xoff, Points[Indices[i]].y*Scale+240+Yoff, mousex*Scale+320+Xoff, mousey*Scale+240+Yoff, Color);
               }
               else PD.Line(FakeBack, Points[Indices[0]].x*Scale+320+Xoff, Points[Indices[0]].y*Scale+240+Yoff, mousex*Scale+320+Xoff, mousey*Scale+240+Yoff, Color);
            }
            break;
      }
      myBack = FakeBack;
   }

   // now do the actual blit
   #ifdef WINDOWED
      if (Cgraph.Blit(&Cgraph.Primary, myBack, NULL, NULL) != CDD_NOERROR)
      {
         if (Cgraph.GetLastDDError() == DDERR_SURFACELOST)
         {
            if (Cgraph.RestoreAll() != CDD_NOERROR) return U_FATAL;
            if (Cgraph.Blit(&Cgraph.Primary, myBack, NULL, NULL) != CDD_NOERROR) return U_FATAL;
         }
         else return U_FATAL;
      }
   #else
      Cgraph.WaitForRetrace();
      if (Cgraph.BlitFast(&Cgraph.Primary, myBack, 0, 0, NULL) != CDD_NOERROR)
      {
         if (Cgraph.GetLastDDError() == DDERR_SURFACELOST)
         {
            if (Cgraph.RestoreAll() != CDD_NOERROR) return U_FATAL;
            if (Cgraph.BlitFast(&Cgraph.Primary, myBack, 0, 0, NULL) != CDD_NOERROR) return U_FATAL;
         }
         else return U_FATAL;
      }
   #endif

   if (!InOp) ShowCursor(TRUE);

   return U_OK;
} /* UpdateScreen */


int LoadGameData()
{
   if (Cgraph.MakeSurface(DISPLAYWIDTH, DISPLAYHEIGHT, DISPLAYDEPTH,
                          &BackBuffer, CDD_SYSTEM) != CDD_NOERROR ||
       Cgraph.MakeSurface(DISPLAYWIDTH, DISPLAYHEIGHT, DISPLAYDEPTH,
                          &FakeBack, CDD_SYSTEM)   != CDD_NOERROR)
   {
      MsgBox("Error creating back buffer.");
      return 1;
   }
   Cgraph.ClearSurface(BackBuffer, NULL, MYRGB(0, 0, 0));
   Cgraph.UpdateWindowed();
   
   SetWindowText(Child[STC_STATUS], "Press F1 for help.");
   KeyPress c;
   c.Down = TRUE;
   c.Key = DIK_COMMA;
   KeyHistory.push_front(c);
   c.Key = DIK_1;
   KeyHistory.push_front(c);

   PD.SetCDDraw(&Cgraph);
   Cgraph.GetDD()->FlipToGDISurface();

   Objs[0] = new sObject;

   ZeroMemory(&Ofn, sizeof(Ofn));
   Ofn.lStructSize = sizeof(Ofn);
   Ofn.hwndOwner = g_hWnd;
   Ofn.lpstrFilter = "Data Files (*.dat)\0*.dat\0";
   Ofn.nFilterIndex = 1;
   Ofn.nMaxFile = 255;

   return 0;
}


int UnloadGameData()
{
   return 0;
}

   
int MainGameLoop()
{
   MSG msg;
   LARGE_INTEGER LI;
   __int64 tFreq, tLast, tCur, tNext, tInc;
   Status state;
   bool ScreenUpdated = false;

   if (!QueryPerformanceFrequency(&LI))
   {
      MsgBox("No High-Frequency timer supported.");
      return 1;
   }

   if (LoadGameData()) return 1;

   tFreq = LI.QuadPart;
   tInc  = tFreq / VFPS;

   QueryPerformanceCounter(&LI);
   tLast = LI.QuadPart;
   tNext = tLast + tInc;

   while(TRUE)
   {
      if (PeekMessage(&msg, g_hWnd, 0, 0, PM_REMOVE))
      {
         TranslateMessage(&msg);
         DispatchMessage(&msg);
      }

      if (g_bQuitApp) return UnloadGameData();

      if (g_bFocus)
      {
         QueryPerformanceCounter(&LI);
         tCur = LI.QuadPart;
         while (tCur >= tNext)
         {
            state = UpdateGameState();
            switch(state)
            {
               case G_UPDATED:
                  ScreenUpdated = true; // need to redraw the screen
                  break;

               case G_QUIT:
                  SendMessage(g_hWnd, WM_CLOSE, 0, 0);
                  break;
            }
            tNext += tInc;
         }
         if (ScreenUpdated)
         {
            state = UpdateScreen();
            ScreenUpdated = false;
            switch(state)
            {
               case U_FATAL:
                  SendMessage(g_hWnd, WM_CLOSE, 0, 0);
                  break;
            }
         }
      }
      else
      {
         g_bNeedRedraw = true;
         QueryPerformanceCounter(&LI);
         tLast = LI.QuadPart;
         tNext = tLast + tInc;
      }
   }
} /* MainGameLoop */
